from .util import httphelper as web

ERROR_INCORRECT = 'incorrect'
ERROR_NOCONNECT = 'noconnect'
ERROR_NOPARAM = 'noparam'
ERROR_TIMEOUT = 'timeout'

# custom error codes for failed requests
ERROR_FAIL = 'fail'
ERROR_NO_SESSION = 'no_session'
ERROR_INVALID = 'response_invalid'

FILM_TYPE_FILM = 'film'
FILM_TYPE_ANIMATION = 'animation'
FILM_TYPE_SERIES = 'serial'

FILM_ORDER_POPULARITY = 'popularity'
FILM_ORDER_AIR_DATE = 'air_date'
FILM_ORDER_ENCODE = 'encode_date'
FILM_ORDER_3MVIEWS = '3month_views'
FILM_ORDER_KP = 'rating_kp'
FILM_ORDER_IMDB = 'rating_imdb'


class TTVApi:
    _api_base_url = 'http://api.ttvrun.one'

    def __init__(self, login, password, guid=None, session=None):
        self._login = login
        self._password = password
        self._guid = guid
        self._session = session
        self._balance = 0

    @property
    def logged_in(self):
        return self._session not in (None, '')

    @property
    def balance(self):
        """Returns user balance"""
        return self._balance

    @property
    def session(self):
        """Returns current session if logged in, None otherwise"""
        return self._session

    @staticmethod
    def _api_error(e_type=ERROR_FAIL):
        """Returns error result stub"""
        return {'success': 0, 'error': e_type}

    @staticmethod
    def result_ok(result):
        """Basic api call result validation"""
        if result is not None:
            try:
                return bool(int(result.get('success', '0')))
            except ValueError as e:
                return False
        return False

    def request(self, request, data=None, with_session=True):
        """Sends TTV API request"""
        if data is None:
            data = {}
        if with_session:
            if not self._session:
                # with_session but no session? That's an error right here
                return self._api_error(ERROR_NO_SESSION)
            data['session'] = self._session

        res = web.get_json(self._api_base_url + request, data)
        return res if res is not None else self._api_error(ERROR_FAIL)

    def auth(self):
        if not self._password or not self._login:
            return self._api_error(ERROR_INCORRECT)

        res = self.request('/v3/auth.php', {
                               'username': self._login,
                               'password': self._password,
                               'application': 'xbmc',
                               'guid': self._guid
                            }, False)
        if res['success'] == 1:
            self._balance = res['balance']
            self._session = res['session']
        else:
            if res['error'] != ERROR_FAIL:
                self._session = None
        return res

    def _auth_request(self, request, data=None):
        """Internal request wrapper with forced authorization"""
        res = self.request(request, data)
        if res['success'] == 0 or res['error'] != '':
            if self.auth()['success'] != 0:
                res = self.request(request, data)
        return res

    def get_film_genres(self):
        return self._auth_request('/v3/film_genres.php')

    def get_film_alt_genres(self):
        return self._auth_request('/v3/film_alt_genres.php')

    def get_film_alt_list(self, film_type=FILM_TYPE_FILM, genre_id=0, list_order=FILM_ORDER_POPULARITY, skip=0, limit=10):
        data = {
            'type': film_type,
            'order': list_order,
            'skip': skip,
            'limit': limit
        }
        if int(genre_id) > 0:
            data['genre_id'] = genre_id
        return self._auth_request('/v3/film_alt_list.php', data)

    def get_film_info(self, film_id):
        return self._auth_request('/v3/film_info.php', {
                                    'film_id': film_id
                                })

    def get_film_alt_info(self, film_id):
        return self._auth_request('/v3/film_alt_info.php', {
                                    'film_id': film_id
                                })

    def get_film_alt_http(self, film_id):
        return self._auth_request("/v3/film_alt_http.php", {
                                    'film_id': film_id
                                })

    def film_alt_search(self, query):
        if len(query) > 255:
            query = query[:254]
        return self._auth_request('/v3/film_alt_search.php', {
                                    'q': query
                                })

    def get_film_posters(self, film_id):
        return self._auth_request("/v3/film_posters.php", {
                                    'film_id': film_id
                                })

    def get_film_reviews(self, film_id):
        return self._auth_request("/v3/film_reviews.php", {
                                    'film_id': film_id
                                })

    def get_film_facts(self, film_id):
        return self._auth_request("/v3/film_facts.php", {
                                    'film_id': film_id
                                })

    def get_film_trailers(self, film_id):
        return self._auth_request("/v3/film_trailers.php", {
                                    'film_id': film_id
                                })

    def get_user_info(self):
        return self._auth_request("/v3/userinfo.php")
